"use client"

import { cn } from "@/lib/utils";
import { ComponentWithClassName } from "@/types/utils";
import { ChevronLeftIcon, ChevronRightIcon } from "@radix-ui/react-icons";
import { flexRender, getCoreRowModel, RowData, TableOptions, useReactTable } from "@tanstack/react-table";
import { Button } from "./ui/button";
import { Skeleton } from "./ui/skeleton";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "./ui/table";

export function LoadingTable<TData extends RowData>({
  className,
  columns,
  rowCount = 5,
  overrides,
}: ComponentWithClassName<{
  columns: TableOptions<TData>["columns"],
  rowCount?: number,
  overrides?: {
    footer?: boolean,
    header?: boolean,
  }
}>) {
  const table = useReactTable({
    data: [],
    columns,
    getCoreRowModel: getCoreRowModel(),
    enableSorting: false,
    enableColumnFilters: false,
    enableColumnPinning: false,
    enableColumnResizing: false,
    enableExpanding: false,
    enableFilters: false,
    enableGlobalFilter: false,
    enableGrouping: false,
    enableHiding: false,
    enableMultiRemove: false,
    enableMultiRowSelection: false,
    enableMultiSort: false,
    enableRowPinning: false,
    enableRowSelection: false,
    enableSortingRemoval: false,
    enableSubRowSelection: false,
    enablePinning: false,
  })
  const showFooter = typeof overrides?.footer === 'undefined' ? true : !!overrides.footer;
  const showHeader = typeof overrides?.header === 'undefined' ? true : !!overrides.header;
  const headerGroups = table.getHeaderGroups();
  const columnCount = headerGroups.reduce<number>((total, current) => total + current.headers.length, 0);
  return (
    <div className={cn(
      "grid gap-3",
      showFooter ? "" : "pb-6",
      className
    )}>
      <div className="grid gap-8">
        <div className={"overflow-x-auto"}>
          <div>
            <div className="rounded-md border">
              <Table className="min-w-[64rem]">
                {showHeader && (
                  <TableHeader>
                    {headerGroups.map((headerGroup) => (
                      <TableRow key={headerGroup.id}>
                        {headerGroup.headers.map((header) => {
                          return (
                            <TableHead
                              className="[&:nth-child(2)]:pl-4 bg-background"
                              key={header.id}
                            >
                              {header.isPlaceholder
                                ? null
                                : flexRender(
                                  header.column.columnDef.header,
                                  header.getContext()
                                )}
                            </TableHead>
                          )
                        })}
                      </TableRow>
                    ))}
                  </TableHeader>
                )}

                <TableBody>
                  {rowCount > 0 ? (
                    [...Array.from(Array(rowCount))].map((_, i) => (
                      <TableRow
                        key={i}
                        className="[&>td:first-child]:pl-6"
                      >
                        {[...Array.from(Array(columnCount))].map((_, ii) => (
                          <TableCell
                            key={ii}
                            className="py-4 [&:nth-child(2)]:pl-4 last-of-type:max-w-none bg-background"
                          >
                            <Skeleton className="w-full h-4" />
                          </TableCell>
                        ))}
                      </TableRow>
                    ))
                  ) : (
                    <TableRow>
                      <TableCell colSpan={columns.length} className="h-24 text-center">
                        <div
                          className={"w-full h-full border border-dashed rounded place-content-center border-border"}
                        >
                          <p className="text-xs text-muted-foreground text-center">
                            No results.
                          </p>
                        </div>
                      </TableCell>
                    </TableRow>
                  )}
                </TableBody>
              </Table>
            </div>

          </div>
        </div>
      </div>

      {!!showFooter && (
        <div className="flex items-center justify-end space-x-2 py-4 p-6 pt-0">
          <div className="flex-1 text-sm text-muted-foreground">
            Showing{" "}
            1 of{" "}
            1 page(s).
          </div>

          <Button
            variant="ghost"
            size="icon"
            onClick={() => table.firstPage()}
            disabled={!table.getCanPreviousPage()}
          >
            <div className="flex flex-row -space-x-2">
              <ChevronLeftIcon className="size-4" />
              <ChevronLeftIcon className="size-4" />
            </div>
          </Button>
          <Button
            variant="ghost"
            size="icon"
            onClick={() => table.previousPage()}
            disabled={!table.getCanPreviousPage()}
          >
            <div className="flex flex-row -space-x-2">
              <ChevronLeftIcon className="size-4" />
            </div>
          </Button>
          <Button
            variant="ghost"
            size="icon"
            onClick={() => table.nextPage()}
            disabled={!table.getCanNextPage()}
          >
            <div className="flex flex-row -space-x-2">
              <ChevronRightIcon className="size-4" />
            </div>
          </Button>
          <Button
            variant="ghost"
            size="icon"
            onClick={() => table.lastPage()}
            disabled={!table.getCanNextPage()}
          >
            <div className="flex flex-row -space-x-2">
              <ChevronRightIcon className="size-4" />
              <ChevronRightIcon className="size-4" />
            </div>
          </Button>
        </div>
      )}
    </div>
  )
}
